/*
 *  BGCore
 *  
 *  Sources may not be modified, distributed, copied, or compiled
 *  in partial or in full, without explicit written approval from
 *  Bight Interactive Inc.
 *
 *  Copyright 2006-2011 Bight Interactive Inc. All rights reserved.
 *
 */


precision mediump float;
precision mediump int;

varying vec2 v_texCoord;
varying vec2 v_texCoord2;
varying vec2 v_Depth;

#ifdef MIPMAPPING
const float mipMapBias = -1.0;
#endif

uniform lowp vec4 ambientColour;

#ifdef DIFFUSETEXTURE
uniform lowp sampler2D diffuseTexture;
#endif

#ifdef BLENDTEXTURE
uniform lowp sampler2D blendTexture;
#endif

#ifdef DIFFUSEVERTEX
varying lowp vec4 v_vertexColour;
#endif

#ifdef DIFFUSEUNIFORM
uniform lowp vec4 diffuseColour;
#endif

#ifdef NORMALTEXTURE
uniform sampler2D normalTexture;
#endif 

#ifdef ALPHA_TEST
uniform float gAlphaTestVal;
#endif 

#if (defined(NORMALTEXTURE) || defined(NORMALVERTEX)) && !defined(CONSTANT)
varying vec3 v_lightvec;
varying lowp vec3 v_normals;
#endif

#if defined(SPECULAR) || defined(BLINN)
varying vec3 v_viewDiff;
varying vec3 v_camPos;
uniform float shininess;
#endif

float Attenuation(float lfDistance, float lfRange)
{
    return lfRange / (lfDistance * lfDistance);
}

lowp vec4 GetDiffuseColour()
{
	lowp vec4 lColour; 
	#ifdef DIFFUSETEXTURE
		
		// Get the colour from the texture
		#ifdef MIPMAPPING
			lColour = texture2D(diffuseTexture, v_texCoord, mipMapBias);
			#ifdef USING_SINGLE_COMPONENT_DIFFUSE_TEXTURE
				lColour.rgb = lowp vec3(1.0, 1.0, 1.0);
			#endif
		#else
			lColour = texture2D(diffuseTexture, v_texCoord);
			#ifdef USING_SINGLE_COMPONENT_DIFFUSE_TEXTURE
				lColour.rgb = lowp vec3(1.0, 1.0, 1.0);
			#endif
		#endif
		
		#ifdef DIFFUSEVERTEX
			lColour *= v_vertexColour;
		#endif
		#ifdef DIFFUSEUNIFORM
			lColour *= diffuseColour;
		#endif
	#elif defined(DIFFUSEVERTEX)
		lColour = v_vertexColour;
	#elif defined(DIFFUSEUNIFORM)
		lColour = diffuseColour;
	#else
		lColour = lowp vec4(1.0, 1.0, 1.0, 1.0);
	#endif
	return lColour;
}

#if defined(BLENDTEXTURE)
lowp vec4 GetBlendColour()
{
	lowp vec4 lBlendColour;
	#if defined(UVS2VERTEX)
		lBlendColour = texture2D(blendTexture, v_texCoord2);
	#else
		#ifdef DIFFUSETEXTURE
			lBlendColour = texture2D(blendTexture, v_texCoord);
		#else
			lBlendColour = texture2D(blendTexture, v_texCoord);
			lBlendColour.rgb = lowp vec3(1.0, 1.0, 1.0);
		#endif
	#endif
	#ifdef USING_SINGLE_COMPONENT_BLEND_TEXTURE
		lBlendColour.rgb = lowp vec3(1.0, 1.0, 1.0);
	#endif
		
	return lBlendColour;
}
#endif

#if defined(NORMALTEXTURE) || defined(NORMALVERTEX) && !defined(CONSTANT)
lowp vec3 GetNormal()
{
	#if defined(NORMALTEXTURE)
		return texture2D(normalTexture, v_texCoord).rgb * 2.0 - 1.0;
	#elif defined(NORMALVERTEX)
		return v_normals;
	#endif
}
#endif

float GetDiffuse(lowp vec3 lNormal, lowp vec3 lLightVec)
{
	float lDiffuse = max(dot(lNormal, lLightVec), 0.0);
	return lDiffuse;
}

#if defined(SPECULAR)
	
float GetSpecular(lowp vec3 lNormal, lowp vec3 lLightVec)
{
	float lSpecular;

	lowp vec3 lHalf = normalize(lLightVec + v_viewDiff);
	lSpecular = pow(max(dot(lNormal, lHalf), 0.0), shininess);
	
	return lSpecular;
}

#endif

#if defined(BLINN)
float GetBlinn(lowp vec3 lNormal, lowp vec3 lLightVec)
{
	float lSpecular;
	
	lowp vec3 lHalf = normalize(lLightVec + v_viewDiff);
	lSpecular = pow(max(dot(lHalf, lNormal), 0.0), 2.0);

	return lSpecular;
}
#endif

#if defined(MAX_POINT_LIGHTS)

varying vec3 v_PointLightDiff[MAX_POINT_LIGHTS];
uniform vec3 v_PointLightColour[MAX_POINT_LIGHTS];
uniform int lnNumPointLights;

vec3 ProcessPointLights(vec3 lNormal)
{
	vec3 lOutputColour = vec3(0.0, 0.0, 0.0);
	
	vec3 lLightVec = normalize(v_PointLightDiff[0]);
	float lDist = length(v_PointLightDiff[0]);
	float lDiffuse = GetDiffuse(lNormal, lLightVec);
	float lAttenuation = Attenuation(lDist, 3.0);
	
	lOutputColour += lDiffuse * v_PointLightColour[0] * lAttenuation;
	
	return lOutputColour;
}

#endif

#if defined(MAX_SPOT_LIGHTS)

varying vec3	v_SpotLightDiff[MAX_SPOT_LIGHTS];
uniform vec3	v_SpotLightDir[MAX_SPOT_LIGHTS];
uniform float	lnSpotLightTheta[MAX_SPOT_LIGHTS];
uniform vec3	v_SpotLightColour[MAX_SPOT_LIGHTS];
uniform int		lnNumSpotLights;

vec3 ProcessSpotLights(vec3 lNormal)
{
	vec3 lOutputColour = vec3(0.0, 0.0, 0.0);
	
	vec3 lLightVec = normalize(v_SpotLightDiff[0]);
	float lDist = length(v_SpotLightDiff[0]);
	float lDiffuse = GetDiffuse(lNormal, lLightVec);

	float spotDot = dot(v_SpotLightDir[0], lLightVec);
	float spotEffect = (spotDot < lnSpotLightTheta[0]) ? 0.0 : lDiffuse;
		
	float lAttenuation = spotEffect <= 0.0 ? 0.0 : 1.0 - (lnSpotLightTheta[0] - spotDot) / (1.0 - lnSpotLightTheta[0]);
		
	lOutputColour += spotEffect * v_SpotLightColour[0] * lAttenuation;

	return lOutputColour;
}

#endif
void main()
{
	lowp vec4 lDiffuseColour = GetDiffuseColour();
	
	#if defined (ALPHA_TEST)
	if(lDiffuseColour.a < gAlphaTestVal)
		discard;
	#endif
 
	#if defined(SPECULAR) || defined(BLINN) || defined (LAMBERT)
	lowp vec3 lNormal = GetNormal();
	#endif

	#ifdef BLENDTEXTURE
	lDiffuseColour *= GetBlendColour();
	#endif
	
	#if defined (LAMBERT) 
		gl_FragColor = lDiffuseColour;
		gl_FragColor.xyz *= (GetDiffuse(lNormal, v_lightvec) + ambientColour.xyz);
		
		#if defined(MAX_POINT_LIGHTS)
		gl_FragColor.xyz += lDiffuseColour.xyz * ProcessPointLights(lNormal);
		#endif
		
		#if defined(MAX_SPOT_LIGHTS)
		gl_FragColor.xyz += lDiffuseColour.xyz * ProcessSpotLights(lNormal);
		#endif
		
	#elif defined(SPECULAR) || defined(BLINN)
		
		// Get the specular for a directional light
		#if defined(SPECULAR)
			float lSpecular = GetSpecular(lNormal, v_lightvec);
		#else
			float lSpecular = GetBlinn(lNormal, v_lightvec);
		#endif
		
		gl_FragColor = lDiffuseColour;
		gl_FragColor.xyz *= (lSpecular + GetDiffuse(lNormal, v_lightvec) + ambientColour.xyz);
		
		#if defined(MAX_POINT_LIGHTS)
		gl_FragColor.xyz += lDiffuseColour.xyz * ProcessPointLights(lNormal);
		#endif
		
		#if defined(MAX_SPOT_LIGHTS)
		gl_FragColor.xyz += lDiffuseColour.xyz * ProcessSpotLights(lNormal);
		#endif
		
	#else
		gl_FragColor = lDiffuseColour;
	#endif	
	//gl_FragColor.x = v_Depth.x;
	//gl_FragColor.y = v_Depth.x;
	//gl_FragColor.z = v_Depth.x;
}
